/**
 * \file: svg_font.c
 *
 * \version: $Id: svg_font.c,v 1.43 2011/06/27 08:44:08 efriedrich Exp $
 *
 * \component:  svg_font for displaying texts
 *
 * \author:
 *
 * \copyright: (c) 2003 - 2004 ADIT Corporation
 *
 * \history
 *
 ***********************************************************************/


#include "svg_typedef.h"
#include "svg_error.h"
#include "svg_font_err.h"
#include "grl_font.h"
#include "svg_font.h"
#include "grl_dlt_log.h"
#include "grl_os_abstraction.h"
#include "EGL/egl.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <semaphore.h>
#include <sys/mman.h>
#include <unistd.h>

#ifdef SVG_FONT_THREAD_SAFE
#define DSNAME_FN1 "ADSVGfn1"
static GRL_lock_id GRL_COMMON_LOCK_ID_FONT;
#   define  SVG_FONT_WAI_LOCK    GRL_wai_lock(GRL_COMMON_LOCK_ID_FONT);
#   define  SVG_FONT_SIG_LOCK    GRL_sig_lock(GRL_COMMON_LOCK_ID_FONT);
#else
#   define  SVG_FONT_WAI_LOCK
#   define  SVG_FONT_SIG_LOCK
#endif

/****************************** Global Params ********************************/

/* Param to signal that plugins are initialized */
static SVGBoolean                    g_is_initialized   = SVG_FALSE;
/* Param for error handling of functions without font contextes */
static SVGError                      g_error            = SVG_NO_ERROR;
/* Global start pointer of module info list */
grl_module_info_list                 *gp_SVG_beginModuleInfoList  = NULL;
/* Global start pointer of info pointer list for multi tasking */
static grl_module_info_pointer_list  *gp_SVG_beginInfoPointerList = NULL;



typedef struct
{
	SVGUint32    app_cnt;
}shm_struct;

static shm_struct *gp_svgfont_appsh = NULL;

extern sem_t* GRL_FONT_LOCK_ID;

/* Dlt specific variables for ctx and debug level */
void *grl_sfnt_dlt_ctx = NULL;
SVGUint8 grl_sfnt_debugLevel = SVG_LOG_FATAL;

/************************** Function declaration *****************************/

/* functions; not externaly visible */
static SVGBoolean local_is_valid_ctx(SVGFontContext* ctx, SVGInt32 function);
static SVGBoolean local_create_module(SVGFontModule module);
static void local_destroy_modules(void);
static SVGBoolean local_create_module_pointer(grl_module_info_list** pp_list);
static void local_destroy_module_pointer(void);
static SVGBoolean local_get_module_pointer(grl_module_info_list** pp_list);
static SVGBoolean local_set_module_pointer(grl_module_info_list* p_list);
static void svgSetFontError(SVGFontContext* ctx, SVGError error);
extern void x86_GRL_font_enter (void);
extern void x86_GRL_font_exit (void);

/* dlt functions declaration */
void sfnt_dlt_SetDebugLevel(SVGUint8 Level );
void sfnt_dlt_GetDebugLevel( SVGUint8 *DebugLevel  );
SVGInt32 sfntDltLogLevelInjection(SVGUint32 service_id, void * data, SVGUint32 length);

/********************************** API **************************************/

static void svgSetFontError(SVGFontContext* ctx, SVGError error)
{
    if (ctx == NULL)
    {
        if (g_error == SVG_NO_ERROR)
        {
            g_error = error;
        }
    }
    else
    {
        if (ctx->error == SVG_NO_ERROR)
        {
            ctx->error = error;
        }
    }
}

SVGError svgGetFontError(SVGFontContext* p_ctx)
{
    SVGError       ret = SVG_NO_ERROR;

    if (NULL == p_ctx)
    {
        /* get error when no context is created */
        ret = g_error;
        g_error = SVG_NO_ERROR;
    }
    else
    {
        ret = p_ctx->p_grl_fp_table->GetFontError(p_ctx);
    }

    return ret;
}


/*******************************************************************************
**  sfnt_dlt_SetDebugLevel
**  Set the debug level.
**
**  INPUT:
**      SVGUint32 Level
**          New debug level.
**  OUTPUT:
**      Nothing.
*/

void sfnt_dlt_SetDebugLevel(SVGUint8 Level )
{
	grl_sfnt_debugLevel = Level;
}

/*******************************************************************************
**  sfnt_dlt_GetDebugLevel
**  Get the current debug level.
**
**  INPUT:
**      Nothing.
**  OUTPUT:
**      SVGUint32 DebugLevel
**          Handle to store the debug level.
*/
void sfnt_dlt_GetDebugLevel( SVGUint8 *DebugLevel  )
{
    *DebugLevel = grl_sfnt_debugLevel;
}

SVGInt32 sfntDltLogLevelInjection(SVGUint32 service_id, void * data,
					SVGUint32 length )
{
	SVGUint32 i = 0;
	char* buf = (char *)data;

	UNUSED_ARG(length);

	SVGUint8 newDebugLevel = SVG_LOG_FATAL;

	sfnt_dlt_GetDebugLevel(&newDebugLevel);

	for (i = 0 ; buf[i]; i++)
		buf[i] = tolower(buf[i]);
	if (strncmp("fatal", buf, 5) == 0)
		newDebugLevel = SVG_LOG_FATAL;
	else if (strncmp("error", buf, 5) == 0)
		newDebugLevel = SVG_LOG_ERROR;
	else if (strncmp("warning", buf, 7) == 0)
		newDebugLevel = SVG_LOG_WARN;
	else if (strncmp("info", buf, 4) == 0)
		newDebugLevel = SVG_LOG_INFO;
	else if (strncmp("debug", buf, 5) == 0)
		newDebugLevel = SVG_LOG_DEBUG;
	else if (strncmp("verbose", buf, 7) == 0)
		newDebugLevel = SVG_LOG_VERBOSE;

	sfnt_dlt_SetDebugLevel(newDebugLevel);

	SVG_FNT_U("Log level set to %u for service ID %x ", newDebugLevel,service_id);

	return 0;
}

void svgInitFonts(void)
{
    SVGFontModule       module = {0, "\0", NULL};

    /* initialize only when never done before */
    if (SVG_FALSE == g_is_initialized)
    {
    	g_is_initialized = SVG_TRUE;
        /* font list begins at NULL */
        gp_SVG_beginModuleInfoList = NULL;

    	grl_sfnt_dlt_ctx = svg_init_dlt("SFNT", "DLT CONTEXT CREATED FOR SVG FONT",
				DLT_SERVICE_LOGLVL_SFNT,
				sfntDltLogLevelInjection );

		SVG_FNT_D("ENTERED INTO SVGINITFONTS");


        /********  Initial font modules  ********/

        /* call plugin modules init */
        grlft_init_font_plugin( &module );
        /* Reading shaping engine configuration */
        grlftReadShpEngConfig();
        if (local_create_module( module ) != SVG_TRUE)
        {
/* PRQA: 3122: magic string is easier */
/*PRQA S 3122 L1 */
            SVG_FNT_E("MODULE_INIT_FAILED  in SVGINITFONTS");
            g_is_initialized = SVG_FALSE;
        }
        else
        {
            /* buildup font database only when modul creation was successfull */
            if (GRL_buildup_font_database() != SVG_TRUE)
            {
                SVG_FNT_E(
                		"BUILDING_FONT_DATABASE_FAILED  in SVGINITFONTS");
                local_destroy_modules();
                g_is_initialized = SVG_FALSE;
            }
            else
            {
#ifdef SVG_FONT_THREAD_SAFE
                if(GRL_NO_ERROR == GRL_cre_lock(&GRL_COMMON_LOCK_ID_FONT, DSNAME_FN1))
                {
                    g_is_initialized = SVG_TRUE;
                }
                else
                {
                    SVG_FNT_E(
                    		"CREATING_SEMAPHORE_FAILED  in SVGINITFONTS");
                    local_destroy_modules();
                    g_is_initialized = SVG_FALSE;
                }
#endif
            }
        }
    }
    else
    {
        SVG_FNT_W("SVGINITFONTS ALREADY_INITIALIZED");
    }
    SVG_FNT_D("EXITING FROM SVGINITFONTS");
/*PRQA L L1 */
}

void svgUpdateFonts(void)
{
    SVG_FNT_D("ENTERED INTO SVGUPDATEFONTS");
    /* initialize only when never done before */
    if (SVG_TRUE == g_is_initialized)
    {
        SVG_FONT_WAI_LOCK
        g_is_initialized = SVG_FALSE;
        local_destroy_modules();
        local_destroy_module_pointer();
        SVG_FONT_SIG_LOCK
        svgInitFonts();

    }
    SVG_FNT_D("EXITING FROM SVGUPDATEFONTS");
}

void svgCloseFonts(void)
{
    SVG_FNT_D("ENTERED INTO SVGCLOSEFONTS");
     /*lock font shutdown*/
    SVG_FONT_WAI_LOCK
    if (SVG_TRUE == g_is_initialized)
    {
        grlft_close_font_plugin();
        local_destroy_modules();
        local_destroy_module_pointer();
        g_is_initialized = SVG_FALSE;
    }
    else
    {
/* PRQA: 3122: magic string is easier */
/*PRQA S 3122 L1 */
        SVG_FNT_W("SVG_NOT_INITIALIZED IN SVGCLOSEFONTS");
/*PRQA L L1 */
    }
    /* free lock */
    SVG_FONT_SIG_LOCK
    SVG_FNT_D("EXITING FROM SVGCLOSEFONTS");
}

SVGBoolean svgGetAvailableFontModules(SVGFontModule *font_module,
                                      SVGUint32 module_id,
                                      SVGUint32 list_control)
{
    SVGBoolean            ret               = SVG_FALSE;
    SVGBoolean            IsPointerCreated  = SVG_FALSE;
    grl_module_info_list  *p_moduleInfoList = NULL;

    SVG_FNT_D("ENTERED INTO SVGGETAVAILABLEFONTMODULES");

    if (font_module != NULL)
    {
        if (gp_SVG_beginModuleInfoList != NULL)     /* if modules are present */
        {
            SVG_FONT_WAI_LOCK
            if (module_id != 0)                    /* searching for module id */
            {
                p_moduleInfoList = gp_SVG_beginModuleInfoList;

                while ((p_moduleInfoList != NULL) && (ret == SVG_FALSE))
                {
                    if (p_moduleInfoList->p_module_info->p_module->id ==
                        module_id)
                    {
                        ret = SVG_TRUE;
                    }
                    else
                    {
                        p_moduleInfoList = p_moduleInfoList->next;
                    }
                }
            }
            else
            {
                IsPointerCreated =
                    local_get_module_pointer(&p_moduleInfoList);
                if (IsPointerCreated == SVG_FALSE)
                {
                    IsPointerCreated =
                        local_create_module_pointer(&p_moduleInfoList);
                    if (IsPointerCreated != SVG_TRUE)
                    {
                        SVG_FNT_W("CREATE_POINTER_FAILED IN "
                        		"SVGGETAVAILABLEFONTMODULES");
                    }
                }

                switch (list_control)
                {
                    case SVG_TOP :
                        p_moduleInfoList = gp_SVG_beginModuleInfoList;
                        ret = SVG_TRUE;
                        break;
                    case SVG_NEXT:
                        if (p_moduleInfoList->next != NULL)
                        {
                            p_moduleInfoList = p_moduleInfoList->next;
                            ret = SVG_TRUE;
                        }
                        break;
                    default:
                        break;
                }

                /* for MT */
                if (local_set_module_pointer(p_moduleInfoList) != SVG_TRUE)
                {
                    SVG_FNT_W("CREATE_MODULE_INFO_LIST_FAILED IN "
                    		"SVGGETAVAILABLEFONTMODULES");
                }
            }
            SVG_FONT_SIG_LOCK
        }
        else                                        /* no modules are present */
        {
            svgSetFontError(NULL, SVG_NOT_INITIALIZED);
            SVG_FNT_E("SVG_NOT_INITIALIZED IN "
            		"SVGGETAVAILABLEFONTMODULES");
        }

        if ((ret == SVG_TRUE) && (p_moduleInfoList != NULL))
        {
            *font_module = *p_moduleInfoList->p_module_info->p_module;
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_POINTER_NULL);
        SVG_FNT_E("SVG_POINTER_NULL IN SVGGETAVAILABLEFONTMODULES");
    }
	
    SVG_FNT_D("EXITING FROM SVGGETAVAILABLEFONTMODULES");
    return ret;
}

SVGFontContext* svgCreateFontContext(const SVGFontModule    *fontModule,
                                     EGLContext  drawContext,
                                     SVGFontContext * shareContext)
{
    grl_module_info    *p_module_info   = NULL;
    SVGFontContext     *p_ctx           = NULL;
	
    SVG_FNT_D("ENTERED INTO SVGCREATEFONTCONTEXT");

    /* call CreateContext of selected font plugin using provided pointer */
    if ((NULL != fontModule) && (SVG_TRUE == g_is_initialized))
    {
    	if (NULL == drawContext)
    	{
            SVG_FNT_W("Creating font context without EGL context. "
            		"No drawing can be done with this context");
    	}
        if (NULL != fontModule->p_grl_fp_table->CreateFontContext)
        {
            SVG_FONT_WAI_LOCK
            while (GRL_get_next_module_info( &p_module_info ) == SVG_TRUE)
            {
                if(p_module_info->p_module->p_grl_fp_table->GetFontError ==
                    fontModule->p_grl_fp_table->GetFontError)
                {
                    p_ctx = fontModule->p_grl_fp_table->CreateFontContext(drawContext, shareContext);
                    if (p_ctx != NULL)
                    {
                        /* no valid font loaded yet */
                        /* when a shared context is used for initialization do not
                           change the valid setting of the shared context */
                        if (NULL == p_ctx->shared)
                        {
                            p_ctx->validFont = SVG_FALSE;
                        }
                        
                        p_ctx->p_module_info = p_module_info;
						
						p_ctx->opacity = 1.0f;
                        /* set scaling factors to 1 by default */
                        p_ctx->scale_x = 1.0f;
                        p_ctx->scale_y = 1.0f;
                    }
                }
            }
            SVG_FONT_SIG_LOCK
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_POINTER_NULL);
        SVG_FNT_E("SVG_POINTER_NULL IN SVGCREATEFONTCONTEXT");
    }
    
    SVG_FNT_D("EXITING FROM SVGCREATEFONTCONTEXT");
    return (p_ctx);
}


void svgDestroyFontContext(SVGFontContext* ctx)
{
    SVG_FNT_D("ENTERED INTO SVGDESTROYFONTCONTEXT");
    if (NULL != ctx)
    {
        SVG_FONT_WAI_LOCK
        if(SVG_FALSE == ctx->p_grl_fp_table->DestroyFontContext(ctx))
        {
            svgSetFontError(ctx, SVG_FONT_CONTEXT_IN_USE);
            SVG_FNT_E("SVG_FONT_CONTEXT_IN_USE IN SVGDESTROYFONTCONTEXT");
        }
        SVG_FONT_SIG_LOCK
        ctx = NULL;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGDESTROYFONTCONTEXT");
    }
    SVG_FNT_D("EXITING FROM SVGDESTROYFONTCONTEXT");
}

void svgGetUsedFontModule( SVGFontContext *ctx,
                           SVGFontModule        *font_module )
{
    SVG_FNT_D("ENTERED INTO SVGGETUSEDFONTMODULE");
    if ((ctx != NULL) && (font_module != NULL))
    {
        *font_module = *ctx->p_module_info->p_module;
    }
    else
    {
        if(NULL == ctx)
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETUSEDFONTMODULE");
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGGETUSEDFONTMODULE");
        }
    }
    SVG_FNT_D("EXITING FROM SVGGETUSEDFONTMODULE");

}

void svgDumpFontWallpaper(  SVGFontContext* ctx, 
							SVGUint32 		settings, 
							SVGInt8 		*fileName)
{
    SVG_FNT_D("ENTERED INTO SVGDUMPFONTWALLPAPER");
	
    if (local_is_valid_ctx( ctx, SVGDUMPFONTWALLPAPER ) == SVG_TRUE)
    {
		SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->DumpFontWallpaper( ctx,
                                                settings,
                                                fileName);
        SVG_FONT_SIG_LOCK
    }
    
    SVG_FNT_D("EXITING FROM SVGDUMPFONTWALLPAPER");
}


SVGBoolean svgGetAvailableFonts( SVGFontContext     *ctx,
                                 SVGFont            *font,
                                 const SVGUint8     *fontName,
                                 SVGUint32          settings,
                                 SVGUint32           list_control )
{
    SVGBoolean  ret = SVG_FALSE;
    SVGFont     *p_font = NULL;

    SVG_FNT_D("ENTERED INTO SVGGETAVAILABLEFONTS");

    if ((ctx != NULL) && (font != NULL) && (SVG_TRUE == g_is_initialized))
    {
        SVG_FONT_WAI_LOCK
        if ((settings <= SVG_FONT_KERNING) &&
            ((list_control == SVG_NEXT) || (list_control == SVG_TOP)))
        {
            p_font = ctx->p_module_info->p_svgFlashFonts;

            if (list_control == SVG_TOP)
            {
                ctx->p_module_info->flashFontCnt = 0;
                ctx->p_module_info->fileFontCnt = 0;
            }

            if (fontName == NULL)
            {
                if (settings == 0)
                {
                    if (ctx->p_module_info->flashFontCnt <
                        ctx->p_module_info->svgFlashFontsMax)
                    {
                        *font = ctx->p_module_info->
                            p_svgFlashFonts[ctx->p_module_info->flashFontCnt];
                        ctx->p_module_info->flashFontCnt++;
                        ret = SVG_TRUE;
                    }
                    else
                    {
                        if (ctx->p_module_info->fileFontCnt <
                            ctx->p_module_info->svgFileFontsMax)
                        {
                            *font = ctx->p_module_info->
                                p_svgFileFonts[ctx->p_module_info->fileFontCnt];
                            ctx->p_module_info->fileFontCnt++;
                            ret = SVG_TRUE;
                        }
                    }
                }
                else
                {
                    /* find first in flash fonts if not at the end */
                    while ((ctx->p_module_info->flashFontCnt <
                            ctx->p_module_info->svgFlashFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if ((p_font[ctx->p_module_info->flashFontCnt].attributes
                            & settings) == settings)
                        {
                            *font = p_font[ctx->p_module_info->flashFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->flashFontCnt++;
                    }

                    /*
                     * find in file fonts if not at the end or not found in
                     * flash fonts
                     */
                    p_font = ctx->p_module_info->p_svgFileFonts;
                    while ((ctx->p_module_info->fileFontCnt <
                            ctx->p_module_info->svgFileFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if ((p_font[ctx->p_module_info->fileFontCnt].attributes
                            & settings) == settings)
                        {
                            *font = p_font[ctx->p_module_info->fileFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->fileFontCnt++;
                    }
                }
            }
            /* font name not NULL */
            else
            {
                p_font = ctx->p_module_info->p_svgFlashFonts;
                if (settings == 0)
                {
                    /* find first in flash fonts if not at the end */
                    while ((ctx->p_module_info->flashFontCnt <
                            ctx->p_module_info->svgFlashFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if (0 == strcmp((const SVGChar*)fontName,
                            (const SVGChar*)p_font[ctx->p_module_info->flashFontCnt].fontName))
                        {
                            *font = p_font[ctx->p_module_info->flashFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->flashFontCnt++;
                    }

                    /*
                     * find in file fonts if not at the end or not found in
                     * flash fonts
                     */
                    p_font = ctx->p_module_info->p_svgFileFonts;
                    while ((ctx->p_module_info->fileFontCnt <
                            ctx->p_module_info->svgFileFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if (0 == strcmp((const SVGChar*)fontName,
                            (const SVGChar*)p_font[ctx->p_module_info->fileFontCnt].fontName))
                        {
                            *font = p_font[ctx->p_module_info->fileFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->fileFontCnt++;
                    }
                }
                else
                {
                    /* find first in flash fonts if not at the end */
                    while ((ctx->p_module_info->flashFontCnt <
                            ctx->p_module_info->svgFlashFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if ((0 == strcmp((const SVGChar*)fontName,
                          (const SVGChar*)p_font[ctx->p_module_info->flashFontCnt].fontName))
                          && (settings == (settings &
                          p_font[ctx->p_module_info->flashFontCnt].attributes)))
                        {
                            *font = p_font[ctx->p_module_info->flashFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->flashFontCnt++;
                    }

                    /*
                     * find in file fonts if not at the end or not found in
                     * flash fonts
                     */
                    p_font = ctx->p_module_info->p_svgFileFonts;
                    while ((ctx->p_module_info->fileFontCnt <
                            ctx->p_module_info->svgFileFontsMax) &&
                            (ret == SVG_FALSE))
                    {
                        if ((0 == strcmp((const SVGChar*)fontName,
                          (const SVGChar*)p_font[ctx->p_module_info->fileFontCnt].fontName))
                          && (settings == (settings &
                          p_font[ctx->p_module_info->fileFontCnt].attributes)))
                        {
                            *font = p_font[ctx->p_module_info->fileFontCnt];
                            ret = SVG_TRUE;
                        }
                        ctx->p_module_info->fileFontCnt++;
                    }
                }
            }
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGGETAVAILABLEFONTS");
        }
        SVG_FONT_SIG_LOCK
    }
    else
    {
        if (ctx == NULL)
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETAVAILABLEFONTS");
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGGETAVAILABLEFONTS");
        }
    }

    SVG_FNT_D("EXITING FROM SVGGETAVAILABLEFONTS");
	
    return ret;
}

void svgSetFontSize( SVGFontContext    *ctx,
                     SVGUint32          fontSize )
{
    SVGUint32 fSize = fontSize;
    
    SVG_FNT_D("ENTERED INTO SVGSETFONTSIZE");
    
    if (local_is_valid_ctx(ctx, SVGSETFONTSIZE) == SVG_TRUE)
    {
        SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->SetFontSize(ctx, fSize);
        SVG_FONT_SIG_LOCK
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTSIZE");
}

void svgLoadFont( SVGFontContext    *ctx,
                  SVGFont           *font,
                  SVGEncoding       charMap,
                  SVGUint32          fontSize )
{
    SVG_FNT_D("ENTERED INTO SVGLOADFONT");
	
    if (NULL != ctx)
    {
        if (NULL != font)
        {
            ctx->p_grl_fp_table->LoadFont(ctx, font, charMap, fontSize);
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGLOADFONT");
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGLOADFONT");
    }
    SVG_FNT_D("EXITING FROM SVGLOADFONT");
}

void svgEnableFontSettings( SVGFontContext* ctx, SVGUint32 settings)
{
    SVG_FNT_D("ENTERED INTO SVGENABLEFONTSETTINGS");
    if (local_is_valid_ctx(ctx, SVGENABLEFONTSETTINGS) == SVG_TRUE)
    {
        SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->EnableFontSettings(ctx, settings);
        SVG_FONT_SIG_LOCK
    }
    SVG_FNT_D("EXITING FROM SVGENABLEFONTSETTINGS");
}

void svgDisableFontSettings( SVGFontContext* ctx, SVGUint32 settings)
{
    SVG_FNT_D("ENTERED INTO SVGDISABLEFONTSETTINGS");
    if (ctx != NULL)
    {
        SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->DisableFontSettings(ctx, settings);
        SVG_FONT_SIG_LOCK
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGDISABLEFONTSETTINGS");
    }
    SVG_FNT_D("EXITING FROM SVGDISABLEFONTSETTINGS");
}

SVGBoolean svgIsFontSettingEnabled( SVGFontContext* ctx, SVGUint32 setting)
{
    SVGBoolean      ret = SVG_FALSE;

    SVG_FNT_D("ENTERED INTO SVGISFONTSETTINGENABLED");

    if (ctx != NULL)
    {
        ret = ctx->p_grl_fp_table->IsFontSettingEnabled(ctx, setting);
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGISFONTSETTINGENABLED");
        ret = SVG_FALSE;
    }
	
    SVG_FNT_D("EXITING FROM SVGISFONTSETTINGENABLED");
	
    return ret;
}

void svgGetFontBitmap( SVGFontContext   *ctx,
                       SVGUint8         *text,
                       SVGUint32        textLength,
                       SVGUint32        *texture,
                       SVGFloat         *width,
                       SVGFloat         *height )
{
    SVG_FNT_D("ENTERED INTO SVGGETFONTBITMAP");

    if (local_is_valid_ctx(ctx, SVGGETFONTBITMAP) == SVG_TRUE)
    {
        if ((texture != NULL) && (text != NULL) && (width != NULL) && (height != NULL))
        {
            SVG_FONT_WAI_LOCK
            ctx->p_grl_fp_table->GetFontBitmap(ctx, text, textLength, texture, width, height);
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTBITMAP");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTBITMAP");
}

void svgFreeFontBitmapMem( SVGFontContext* ctx, SVGUint32 texture)
{
    SVG_FNT_D("EXITING FROM SVGFREEFONTBITMAPMEM");

    if ((ctx != NULL) && (texture != 0))
    {
        SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->FreeFontBitmapMem(ctx, texture);
        SVG_FONT_SIG_LOCK
    }
    else
    {
        if (ctx == NULL)
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_E("SVG_CONTEXT_NULL IN SVGFREEFONTBITMAPMEM");
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGFREEFONTBITMAPMEM");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGFREEFONTBITMAPMEM");
}

void svgDrawFontString( SVGFontContext *ctx,
                         SVGFloat       positionX,
                         SVGFloat       positionY,
                         SVGUint8       *text,
                         SVGUint32      textLength )
{
    SVG_FNT_D("ENTERED INTO SVGDRAWFONTSTRING");

    if (local_is_valid_ctx(ctx, SVGDRAWFONTSTRING) == SVG_TRUE)
    {
        if (text != NULL)
        {
            SVG_FONT_WAI_LOCK
            ctx->p_grl_fp_table->DrawFontWithCursorInfo(ctx,
                                                        positionX,
                                                        positionY,
                                                        text,
                                                        textLength,
                                                        NULL);
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGDRAWFONTSTRING");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGDRAWFONTSTRING");
}

void svgGetFontBBox( SVGFontContext    *ctx,
                      SVGUint8          *text,
                      SVGUint32         textLength,
                      SVGBBox           *textBBox)
{
    SVG_FNT_D("ENTERED INTO SVGGETFONTBBOX");

    if (local_is_valid_ctx(ctx, SVGGETFONTBBOX) == SVG_TRUE)
    {
        if ((textBBox != NULL) && (text != NULL))
        {
            SVG_FONT_WAI_LOCK
            ctx->p_grl_fp_table->
                GetFontBBox(ctx, text, textLength, textBBox);
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTBBOX");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTBBOX");
}

SVGFloat    svgGetFontLineSpacing(const SVGFontContext* ctx)
{
    SVGFloat   ret = 0.0;

    SVG_FNT_D("ENTERED INTO SVGGETFONTLINESPACING");

    if (NULL != ctx)
    {
        ret = ctx->font_line_space;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTLINESPACING");
    }
	
    SVG_FNT_D("EXITING FROM SVGGETFONTLINESPACING");
	
    return ret;
}

SVGFloat svgGetFontRotate( const SVGFontContext* ctx)
{
    SVGFloat    ret = 0;
	
    SVG_FNT_D("ENTERED INTO SVGGETFONTROTATE");
    
    if (NULL != ctx)
    {
        ret = ctx->angle;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTROTATE");
    }
	
    SVG_FNT_D("EXITING FROM SVGGETFONTROTATE");
    return ret;
}

void svgSetFontRotate( SVGFontContext* ctx, SVGFloat angle)
{
    SVG_FNT_D("ENTERED INTO SVGSETFONTROTATE");
    if (local_is_valid_ctx(ctx, SVGSETFONTROTATE) == SVG_TRUE)
    {
        SVG_FONT_WAI_LOCK
        
        ctx->angle = angle;
		if(ctx->angle < MIN_ANGLE)
		{
			ctx->angle = MIN_ANGLE;
		}
		if( ctx->angle < 0.0f )
		{
			ctx->angle += MAX_ANGLE;
		}

		if(ctx->angle > MAX_ANGLE)
		{
			ctx->angle = MAX_ANGLE;
		}
        
        SVG_FONT_SIG_LOCK
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTROTATE");
}

void svgSetFontDrawingOrigin(SVGFontContext* ctx, SVGOriginType originType)
{
    SVG_FNT_D("ENTERED INTO SVGSETFONTDRAWINGORIGIN");
    if (NULL != ctx)
    {
        if ((originType > SVG_ORIGIN_MIN) &&
            (originType < SVG_ORIGIN_MAX))
        {
            SVG_FONT_WAI_LOCK
            ctx->base_line = originType;
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGSETFONTDRAWINGORIGIN");
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETFONTDRAWINGORIGIN");
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTDRAWINGORIGIN");;
}

SVGOriginType    svgGetFontDrawingOrigin( const SVGFontContext* ctx)
{
    SVGOriginType   ret = SVG_ORIGIN_BASELINE;
	
    SVG_FNT_D("ENTERED INTO SVGGETFONTDRAWINGORIGIN");
    
    if (NULL != ctx)
    {
        ret = ctx->base_line;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTDRAWINGORIGIN");
    }
	
    SVG_FNT_D("EXITING FROM SVGGETFONTDRAWINGORIGIN");
    return ret;
}


void svgSetFontLetterDistance( SVGFontContext* ctx, SVGFloat letterDistance )
{
    SVG_FNT_D("ENTERED INTO SVGSETFONTLETTERDISTANCE");
    if (ctx != NULL)
    {
        if(0.0f <= letterDistance)
        {
            SVG_FONT_WAI_LOCK
            ctx->letter_distance = letterDistance;
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGSETFONTLETTERDISTANCE");
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETFONTLETTERDISTANCE");
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTLETTERDISTANCE");
}

SVGFloat svgGetFontLetterDistance( const SVGFontContext* ctx )
{
    SVGFloat   ret = 0.0;

    SVG_FNT_D("ENTERED INTO SVGGETFONTLETTERDISTANCE");

    if (ctx != NULL)
    {
        ret = ctx->letter_distance;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTLETTERDISTANCE");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTLETTERDISTANCE");

    return ret;
}

void svgDrawFontWithCursorInfo( SVGFontContext *ctx,
                                SVGFloat       positionX,
                                SVGFloat       positionY,
                                SVGUint8       *text,
                                SVGUint32      textLength,
                                SVGPoint       *cursorPosArray )
{
    SVG_FNT_D("ENTERED INTO SVGDRAWFONTWITHCURSORINFO");
	
    if (local_is_valid_ctx( ctx, SVGDRAWFONTWITHCURSORINFO ) == SVG_TRUE)
    {
        if ((text != NULL) && (cursorPosArray != NULL))
        {
            SVG_FONT_WAI_LOCK
            ctx->p_grl_fp_table->DrawFontWithCursorInfo( ctx,
                                                          positionX,
                                                          positionY,
                                                          text,
                                                          textLength,
                                                          cursorPosArray );
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError( ctx, SVG_POINTER_NULL );
            SVG_FNT_E("SVG_POINTER_NULL IN SVGDRAWFONTWITHCURSORINFO");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGDRAWFONTWITHCURSORINFO");
}

SVGUint32 svgGetFontMaxChars(SVGFontContext   *ctx,
                             SVGFloat          max_space_x,
                             SVGFloat          max_space_y,
                             SVGUint8         *text,
                             SVGUint32        textLength)
{
    SVGUint32 ret = 0;
    
    SVG_FNT_D("ENTERED INTO SVGGETFONTMAXCHARS");
    
    if (local_is_valid_ctx(ctx, SVGGETFONTMAXCHARS) == SVG_TRUE)
    {
        if ((0 != textLength) && (0 != max_space_x) && (0 != max_space_y))
        {
            if (text != NULL)
            {
                ret = ctx->p_grl_fp_table->
                    GetFontMaxChars(ctx, max_space_x, max_space_y, text, textLength);
            }
            else
            {
                svgSetFontError(ctx, SVG_POINTER_NULL);
                SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTMAXCHARS");
            }
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGGETFONTMAXCHARS");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTMAXCHARS");
    
    return ret;
}

void svgGetFontInformation ( SVGFontContext *ctx, SVGFontInfo *info )
{
	
    SVG_FNT_D("ENTERED INTO SVGGETFONTINFORMATION");
	
    if (local_is_valid_ctx(ctx, SVGGETFONTINFORMATION) == SVG_TRUE)
    {
        ctx->p_grl_fp_table->GetFontInformation(ctx, info);
    }
    else
    {
        svgSetFontError(NULL, SVG_POINTER_NULL);
        SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTINFORMATION");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTINFORMATION");
}

void svgSetFontOpacity( SVGFontContext* ctx, SVGFloat       opacity)
{
	
    SVG_FNT_D("ENTERED INTO SVGSETFONTOPACITY");
	
    if (NULL != ctx)
    {
         /*check value*/
        if ((opacity > 0.0) && (opacity <= 1.0))
        {
            SVG_FONT_WAI_LOCK
            ctx->opacity = opacity;
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGSETFONTOPACITY");
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETFONTOPACITY ");
    }
    
    SVG_FNT_D("EXITING FROM SVGSETFONTOPACITY");
}

SVGFloat svgGetFontOpacity( SVGFontContext* ctx)
{
    SVGFloat return_val = 1.0;
    
    SVG_FNT_D("ENTERED INTO SVGGETFONTOPACITY");
    
    if (NULL != ctx)
    {
        return_val = ctx->opacity;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTOPACITY");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTOPACITY");
    
    return return_val;
}


void svgDrawFontStringExt( SVGFontContext* ctx,
                           SVGPoint* position,
                           SVGFloat* angle,
                           SVGUint8* glyphs,
                           SVGUint32 glyphLength)
{	
    SVG_FNT_D("ENTERED INTO SVGDRAWFONTSTRINGEXT");
	
    if (local_is_valid_ctx(ctx, SVGDRAWFONTSTRINGEXT) == SVG_TRUE)
    {
        if ((glyphs != NULL) && (position != NULL) && (angle != NULL))
        {
        	if(*angle < MIN_ANGLE)
			{
				*angle = MIN_ANGLE;
			}
			if( *angle < 0.0f )
			{
				*angle += MAX_ANGLE;
			}
	
			if(*angle > MAX_ANGLE)
			{
				*angle = MAX_ANGLE;
			}

			SVG_FONT_WAI_LOCK

            ctx->p_grl_fp_table->DrawFontStringExt(ctx,
                                                   position,
                                                   angle,
                                                   glyphs,
                                                   glyphLength);
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_D("SVG_POINTER_NULL IN SVGDRAWFONTSTRINGEXT");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGDRAWFONTSTRINGEXT");
}
 
void svgSetFontScale( SVGFontContext *ctx,
                          SVGFloat       scale_x,
                          SVGFloat       scale_y )
{
    SVG_FNT_D("ENTERED INTO SVGSETFONTSCALE");

    if ( NULL != ctx )
    {
        /*
         * Check parameters: scaling factor must be in the range (0..1]
         * to avoid exceeding the maximum by the scaling.
         */
        if ( (scale_x > 0.0) && (scale_x <= 1.0) &&
             (scale_y > 0.0) && (scale_y <= 1.0) )
        {
            SVG_FONT_WAI_LOCK
            ctx->scale_x = scale_x;
            ctx->scale_y = scale_y;
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGSETFONTSCALE");
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETFONTSCALE");
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTSCALE");
}


void svgGetFontScale( SVGFontContext *ctx,
                      SVGFloat       *scale_x,
                      SVGFloat       *scale_y )
{
    SVG_FNT_D("ENTERED INTO SVGGETFONTSCALE");
	
    if ((NULL != ctx) &&
        (NULL != scale_x) && (NULL != scale_y) )
    {
        *scale_x = ctx->scale_x;
        *scale_y = ctx->scale_y;
    }
    else
    {
        if (NULL == ctx)
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_D("SVG_CONTEXT_NULL IN SVGGETFONTSCALE");
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_D("SVG_POINTER_NULL IN SVGGETFONTSCALE");
        }
    }
    SVG_FNT_D("EXITING FROM SVGGETFONTSCALE");
}

void svgSetContourColor (SVGFontContext *ctx, SVGUint32     color)
{
    SVG_FNT_D("ENTERED INTO SVGSETCONTOURCOLOR");
	
    if (NULL != ctx)
    {
        ctx->ContourColor = color;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETCONTOURCOLOR");
    }
    
    SVG_FNT_D("EXITING FROM SVGSETCONTOURCOLOR");
}

void svgSetContourWidth (SVGFontContext *ctx, SVGUint32     width)
{
    SVG_FNT_D("ENTERED INTO SVGSETCONTOURWIDTH");
    if (NULL != ctx)
    {
        ctx->ContourWidth = width;
        ctx->WidthSet = SVG_TRUE;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETCONTOURWIDTH");
    }
    SVG_FNT_D("EXITING FROM SVGSETCONTOURWIDTH");
}

void svgSetBodyColor (SVGFontContext *ctx, SVGUint32     color)
{	
    SVG_FNT_D("ENTERED INTO SVGSETBODYCOLOR");

    if (NULL != ctx)
    {
        ctx->BodyColor = color;
        ctx->ColorSet = SVG_TRUE;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETBODYCOLOR");
    }
    
    SVG_FNT_D("EXITING FROM SVGSETBODYCOLOR");
}


SVGUint32 svgGetContourColor (SVGFontContext    *ctx)
{
    SVGUint32 return_value = 0;

    SVG_FNT_D("ENTERED INTO SVGGETCONTOURCOLOR");

    if (NULL != ctx)
    {
        return_value = ctx->ContourColor;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETCONTOURCOLOR");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETCONTOURCOLOR");
    
    return return_value;
}

SVGUint32 svgGetContourWidth (SVGFontContext    *ctx)
{
    SVGUint32 return_value = 0;
    
    SVG_FNT_D("ENTERED INTO SVGGETCONTOURWIDTH");
    
    if (NULL != ctx)
    {
        return_value = ctx->ContourWidth;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETCONTOURWIDTH");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETCONTOURWIDTH");
    
    return return_value;
}

SVGUint32 svgGetBodyColor (SVGFontContext    *ctx)
{
    SVGUint32 return_value = 0;
    
    SVG_FNT_D("ENTERED INTO SVGGETBODYCOLOR");
    
    if (NULL != ctx)
    {
        return_value = ctx->BodyColor;
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETBODYCOLOR");
    }
    
    SVG_FNT_D("EXITING FROM SVGGETBODYCOLOR");
    
    return return_value;
}

void svgSetFontReplacementChar( SVGFontContext *ctx,
                                SVGUint8*      replace,
                                SVGUint32      length )
{
    SVG_FNT_D("ENTERED INTO SVGSETFONTREPLACEMENTCHAR");
    
    if (local_is_valid_ctx(ctx, SVGSETFONTREPLACEMENTCHAR) == SVG_TRUE)
    {
        if ((NULL != replace) && 
            (0 != length) && 
            (sizeof(ctx->replacement_char) >= length))
        {
            ctx->new_replacement_char = SVG_TRUE;
            memcpy(ctx->replacement_char, replace, length);
            ctx->replacement_length = length;
        }
        else
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_E("SVG_CONTEXT_NULL IN SVGSETFONTREPLACEMENTCHAR");
        }
    }
    SVG_FNT_D("EXITING FROM SVGSETFONTREPLACEMENTCHAR");

}                                 


void svgGetFontReplacementChar( SVGFontContext *ctx,
                                SVGUint8*      replace,
                                SVGUint32      *length )
{
    SVG_FNT_D("ENTERED INTO SVGGETFONTREPLACEMENTCHAR");
	
    if (local_is_valid_ctx(ctx, SVGGETFONTREPLACEMENTCHAR) == SVG_TRUE)
    {
        if ((NULL != replace) && 
            (NULL != length) && 
            (ctx->replacement_length <= *length))
        {
            *length = ctx->replacement_length;
            memcpy(replace, ctx->replacement_char, *length);
        }
        else
        {
            svgSetFontError(NULL, SVG_CONTEXT_NULL);
            SVG_FNT_E("SVG_CONTEXT_NULL IN SVGGETFONTREPLACEMENTCHAR");
            if(NULL != length)
            {
                *length = 0;
            }
        }
    }
    else
    {
        if(NULL != length)
        {
            *length = 0;
        }
    }
    SVG_FNT_D("EXITING FROM SVGGETFONTREPLACEMENTCHAR");
}                                       

SVGUint32 svgGetFontBitmapExt( SVGFontContext   *ctx,
                               SVGUint8         **text,
                               SVGUint32        *textLength,
                               SVGUint32	   	textNumber,
                               SVGUint32        *filledTexture,
							   SVGUint32        *strokedTexture,
                               SVGBBox	   	    **filledTextureBox,
                               SVGBBox	   	    **strokedTextureBox,
							   SVGBBox		    **filledGlyphBox,
                               SVGBBox		    **strokedGlyphBox,
							   SVGUint32		*glyphNumber)
{
    SVGUint32 ret = 0;
    SVGError  error = SVG_NO_ERROR;

    SVG_FNT_D("ENTERED INTO SVGGETFONTBITMAPEXT");
    
    if (local_is_valid_ctx(ctx, SVGGETFONTBITMAPEXT) == SVG_TRUE)
    {
		if ((ctx->settings & (SVG_FONT_2COLOR | SVG_FONT_OUTLINE)) != 0)
		{
			if(	(strokedTexture == NULL) && (strokedGlyphBox == NULL) && (strokedTextureBox == NULL) )
			{
				error = SVG_POINTER_NULL;
			}				
		}
		if( ((ctx->settings & SVG_FONT_OUTLINE) != 0) &&
			((ctx->settings & SVG_FONT_2COLOR) == 0))
		{
		}
		else
		{
			if((filledTexture == NULL) && (filledGlyphBox == NULL) && (filledTextureBox == NULL))
			{
				error = SVG_POINTER_NULL;
			}
        }
		if ((text != NULL) && (glyphNumber != NULL) && (textLength != NULL) && (error == SVG_NO_ERROR) )
        {
            SVG_FONT_WAI_LOCK
            ret = ctx->p_grl_fp_table->GetFontBitmapExt(ctx, 
														text, 
														textLength, 
														textNumber, 
														filledTexture, 
														strokedTexture, 
														filledTextureBox, 
														strokedTextureBox, 
														filledGlyphBox, 
														strokedGlyphBox, 
														glyphNumber);
            SVG_FONT_SIG_LOCK
        }
        else
        {
            svgSetFontError(ctx, SVG_POINTER_NULL);
            SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTBITMAPEXT");
        }
    }
    
    SVG_FNT_D("EXITING FROM SVGGETFONTBITMAPEXT");
    return ret;
}

SVGUint32 svgGetFontGlyphCount(SVGFontContext   *ctx,
	                             SVGUint8         *text,
	                             SVGUint32        textLength)
{
    SVGUint32 ret = 0;
    SVGUint32 max_space_x = 0XFFFFFFFD;
    SVGUint32 max_space_y = 0XFFFFFFFD;


    SVG_FNT_D("ENTERED INTO SVGGETFONTGLYPHCOUNT");

    if (local_is_valid_ctx(ctx, SVGGETFONTGLYPHCOUNT) == SVG_TRUE)
    {
        if (0 != textLength)
        {
            if (text != NULL)
            {
                ret = ctx->p_grl_fp_table->
                    GetFontMaxChars(ctx, max_space_x, max_space_y, text, textLength);
            }
            else
            {
                svgSetFontError(ctx, SVG_POINTER_NULL);
                SVG_FNT_E("SVG_POINTER_NULL IN SVGGETFONTGLYPHCOUNT");
            }
        }
        else
        {
            svgSetFontError(ctx, SVG_INVALID_VALUE);
            SVG_FNT_E("SVG_INVALID_VALUE IN SVGGETFONTGLYPHCOUNT");
        }
    }

    SVG_FNT_D("EXITING FROM SVGGETFONTGLYPHCOUNT");

    return ret;
}

/*************************** Internal functions ******************************/

static SVGBoolean local_is_valid_ctx(SVGFontContext* ctx, SVGInt32 function)
{
    SVGBoolean      return_value = SVG_FALSE;

    UNREFED_FONT_PARAM(function)
    if (ctx != NULL)
    {
        if (NULL == ctx->shared)
        {
            if (ctx->validFont == SVG_TRUE)
            {
                return_value = SVG_TRUE;
            }
            else
            {
                svgSetFontError(ctx, SVG_INVALID_OPERATION);
                SVG_FNT_E("SVG_INVALID_OPERATION IN %x ",function);
            }
        }
        else
        {
            if (ctx->shared->validFont == SVG_TRUE)
            {
                return_value = SVG_TRUE;
            }
            else
            {
                svgSetFontError(ctx, SVG_INVALID_OPERATION);
                SVG_FNT_E("SVG_INVALID_OPERATION IN %x ",function);
            }
        }
    }
    else
    {
        svgSetFontError(NULL, SVG_CONTEXT_NULL);
        SVG_FNT_E("SVG_CONTEXT_NULL IN %x ",function);
    }

    return return_value;
}

static SVGBoolean local_create_module(SVGFontModule module)
{
    SVGBoolean              ret               = SVG_TRUE;
    SVGFontModule           *p_module         = NULL;
    grl_module_info         *p_moduleInfo     = NULL;
    grl_module_info_list    *p_moduleInfoList = NULL;
    grl_module_info_list    *p_moduleInfoCnt  = gp_SVG_beginModuleInfoList;

    /* create and copy of FontModule */
    p_module = (SVGFontModule*)GRL_malloc_1D_resource(sizeof(SVGFontModule));
    if (NULL != p_module)
    {
        *p_module = module;

        /* create FontModule informations */
        p_moduleInfo = (grl_module_info*)GRL_malloc_1D_resource(sizeof(grl_module_info));
        if (NULL != p_moduleInfo)
        {
            p_moduleInfo->p_module = p_module;
            p_moduleInfo->p_svgFlashFonts = NULL;
            p_moduleInfo->p_svgFileFonts  = NULL;

            /* create of list entry */
            p_moduleInfoList = (grl_module_info_list*)GRL_malloc_1D_resource(sizeof(grl_module_info_list));
            if (NULL != p_moduleInfoList)
            {
                p_moduleInfoList->next          = NULL;
                p_moduleInfoList->p_module_info = p_moduleInfo;

                /* add to list  */
                if (gp_SVG_beginModuleInfoList == NULL)
                {
                    gp_SVG_beginModuleInfoList = p_moduleInfoList;
                }
                else
                {
                    while (p_moduleInfoCnt->next != NULL)
                    {
                        p_moduleInfoCnt = p_moduleInfoCnt->next;
                    }
                    p_moduleInfoCnt->next = p_moduleInfoList;
                }
            }
            else
            {
/* PRQA: 3122: magic string is easier */
/*PRQA S 3122 L1 */
                SVG_FNT_E("CREATE_MODULE_INFO_FAILED IN GRL_CREATE_FONT_MODULE");

                GRL_free_1D_resource((void*) p_moduleInfo->p_module);
                GRL_free_1D_resource((void*) p_moduleInfo);
                svgSetFontError(NULL, SVG_OUT_OF_MEMORY);
                ret = SVG_FALSE;
            }
        }
        else
        {
            SVG_FNT_E("CREATE_MODULE_INFO_FAILED IN GRL_CREATE_FONT_MODULE");
            GRL_free_1D_resource((void*) p_module);
            svgSetFontError(NULL, SVG_OUT_OF_MEMORY);
            ret = SVG_FALSE;
        }
    }
    else
    {
        SVG_FNT_E("CREATE_MODULE_FAILED IN GRL_CREATE_FONT_MODULE");
        svgSetFontError(NULL, SVG_OUT_OF_MEMORY);
        ret = SVG_FALSE;
    }
/*PRQA L L1 */

    return ret;
}

static void local_destroy_modules(void)
{
    grl_module_info_list    *p_moduleInfoCnt = gp_SVG_beginModuleInfoList;
    grl_module_info_list    *p_moduleNext    = NULL;

    while (p_moduleInfoCnt != NULL)
    {
        /* destroy falsh fonts of module */
        if ( p_moduleInfoCnt->p_module_info->svgFlashFontsMax != 0 )
        {
            GRL_free_1D_resource((void*)p_moduleInfoCnt->p_module_info->p_svgFlashFonts);
        }

        /* destroy file fonts of module */
        if ( p_moduleInfoCnt->p_module_info->svgFileFontsMax != 0 )
        {
            GRL_free_1D_resource((void*)p_moduleInfoCnt->p_module_info->p_svgFileFonts);
        }

        /* destroy module */
        GRL_free_1D_resource((void*) p_moduleInfoCnt->p_module_info->p_module);

        /* destroy module information */
        GRL_free_1D_resource((void*) p_moduleInfoCnt->p_module_info);

        /* destroy module information list */
        p_moduleNext = p_moduleInfoCnt->next;
        GRL_free_1D_resource((void*) p_moduleInfoCnt);
        p_moduleInfoCnt = p_moduleNext;
    }
    gp_SVG_beginModuleInfoList = NULL;
}

static SVGBoolean local_create_module_pointer( grl_module_info_list** pp_list )
{
    SVGBoolean                      ret = SVG_TRUE;

    grl_module_info_pointer_list *p_pp_list = NULL;
    grl_module_info_pointer_list *p_pointerCnt  = gp_SVG_beginInfoPointerList;
    /* create and copy of FontModule */
	p_pp_list = (grl_module_info_pointer_list*)GRL_malloc_1D_resource(sizeof(grl_module_info_pointer_list));
    if (NULL != p_pp_list)
    {
        p_pp_list->next          = NULL;

        p_pp_list->p_module_info = gp_SVG_beginModuleInfoList;

        /* add to list  */
        if (gp_SVG_beginInfoPointerList == NULL)
        {
            gp_SVG_beginInfoPointerList = p_pp_list;
        }
        else
        {
            while (p_pointerCnt->next != NULL)
            {
                p_pointerCnt = p_pointerCnt->next;
            }
            p_pointerCnt->next = p_pp_list;
        }

        *pp_list = p_pp_list->p_module_info;
    }
    else
    {
        ret = SVG_FALSE;
    }

    return ret;
}

static void local_destroy_module_pointer( void )
{
    grl_module_info_pointer_list *p_pointerCnt  = gp_SVG_beginInfoPointerList;
    grl_module_info_pointer_list *p_pointerNext = NULL;

    while (p_pointerCnt != NULL)
    {
        /* destroy modulepp_list */
        p_pointerNext = p_pointerCnt->next;
        GRL_free_1D_resource((void*) p_pointerCnt);
        p_pointerCnt = p_pointerNext;
    }
    gp_SVG_beginInfoPointerList = NULL;
}

static SVGBoolean local_get_module_pointer( grl_module_info_list** pp_list )
{
    SVGBoolean                      ret          = SVG_FALSE;
    grl_module_info_pointer_list   *p_pointerCnt = gp_SVG_beginInfoPointerList;
    if (p_pointerCnt != NULL)
    {
		*pp_list = p_pointerCnt->p_module_info;
		ret = SVG_TRUE;
    }

    return ret;
}

static SVGBoolean local_set_module_pointer( grl_module_info_list* p_list )
{
    SVGBoolean                      ret          = SVG_FALSE;
    grl_module_info_pointer_list   *p_pointerCnt = gp_SVG_beginInfoPointerList;

    if(p_pointerCnt != NULL)
    {
		p_pointerCnt->p_module_info = p_list;
		ret = SVG_TRUE;
    }

    return ret;
}

#ifdef X86SIM

/*
* called when svg_layer.so is loaded
*/
void __attribute__ ((constructor))SVG_font_so_init(void)
{
	x86_GRL_font_enter();
}


/*
* called when svg_layer.so is unloaded
*/
void __attribute__ ((destructor))SVG_font_so_uninit(void)
{

	x86_GRL_font_exit();
}


#endif

void x86_GRL_font_enter (void)
{

    SVGInt32     fp 				= 1;
	SVGBoolean  shm_init   		= SVG_FALSE;
	SVGBoolean  mmap_required   = SVG_FALSE;
	SVGInt32    shm_size 		= 0;
	SVGInt32	grl_error  		= 0;
	void *      mmap_struct 	= NULL;
	struct stat shm_stat 		;


	GRL_FONT_LOCK_ID = sem_open("/svg_font_sem", O_CREAT, 0777, 1);
	if( GRL_FONT_LOCK_ID == SEM_FAILED || GRL_FONT_LOCK_ID == NULL )
	{
        SVG_FNT_E("SEM_OPEN_FAILED IN X86_GRL_FONT_ENTER");
	}
	else
	{
		if(chmod("/dev/shm/sem.svg_font_sem", S_IRWXU | S_IRWXG |S_IRWXO)!= 0)
		{
	        SVG_FNT_E("SEM_CHMOD_FAILED IN X86_GRL_FONT_ENTER");
		}
		grl_error = sem_wait(GRL_FONT_LOCK_ID);
		if (0 == grl_error)
		{
				/*creating shared memory segment*/
			fp = shm_open( "/svg_font_appcnt", O_CREAT | O_RDWR, S_IRUSR | S_IWUSR );
			if ( fp <= -1 )
			{
		        SVG_FNT_E("SHM_OPEN_FAILED IN X86_GRL_FONT_ENTER");
			}
			else
			{
				if(chmod("/dev/shm/svg_font_appcnt", S_IRUSR | S_IWUSR | S_IROTH | S_IWOTH )!= 0)
				{
			        SVG_FNT_E("SHM_CHMOD_FAILED IN X86_GRL_FONT_ENTER");
				}
				/* get state */
				if (fstat(fp, &shm_stat ) != 0)
				{
			        SVG_FNT_E("SHM_STATE_FAILED IN X86_GRL_FONT_ENTER");
				}
				else
				{
					/* calculate shared memory size */
					shm_size = (sizeof(shm_struct));
					shm_size = (shm_size + (sysconf(_SC_PAGESIZE) - 1)) & ~(sysconf(_SC_PAGESIZE) - 1);
					if (shm_stat.st_size == 0)
					{
						shm_init = SVG_FALSE ;
						if ( ftruncate( fp, shm_size ) != 0 )
						{
					        SVG_FNT_E("SHM_SIZE_FAILED IN X86_GRL_FONT_ENTER");
							mmap_required   = SVG_FALSE;
						}
						else
						{
							mmap_required   = SVG_TRUE;
						}
					}
					else
					{
						shm_init = SVG_TRUE;
						mmap_required   = SVG_TRUE;
					}
				}
			}

			if(mmap_required   == SVG_TRUE)
			{
							/* Map these sengments in the process memory space */
				mmap_struct = mmap( NULL, shm_size, PROT_READ | PROT_WRITE, MAP_SHARED, fp, 0 );

				if ( mmap_struct == MAP_FAILED )
				{
			        SVG_FNT_E("SHM_MMAP_FAILED IN X86_GRL_FONT_ENTER");
				}
				else
				{
					gp_svgfont_appsh = (shm_struct *)mmap_struct;
					if (shm_init == SVG_FALSE)
					{
						gp_svgfont_appsh->app_cnt = 0;
						shm_init = SVG_TRUE;
					}
					if (shm_init == SVG_TRUE)
					{
						gp_svgfont_appsh->app_cnt ++;
					}
				}
				close(fp);
			}
			grl_error = sem_post(GRL_FONT_LOCK_ID);
			if(0 != grl_error)
			{
		        SVG_FNT_E("SEM_POST_FAILED IN X86_GRL_FONT_ENTER");
			}
		}
		else
		{
	        SVG_FNT_E("SEM_WAIT_FAILED IN X86_GRL_FONT_ENTER");
		}
	}
}


void x86_GRL_font_exit (void)
{
	SVGInt32	grl_error  		= 0;
	grl_error = sem_wait(GRL_FONT_LOCK_ID);
	if(0 == grl_error)
	{
		if(NULL != gp_svgfont_appsh)
		{
			gp_svgfont_appsh->app_cnt--;
			if(gp_svgfont_appsh->app_cnt == 0)
			{
				shm_unlink("/svg_flashfonts");
				shm_unlink("/svg_filefonts");
				shm_unlink("/svg_font_appcnt");
				sem_unlink("/svg_font_sem");
			}
			else
			{
				grl_error = sem_post(GRL_FONT_LOCK_ID);
				if(0 != grl_error)
				{
			        SVG_FNT_E("SEM_POST_FAILED IN X86_GRL_FONT_EXIT");
				}
			}
		}
	}
	else
	{
        SVG_FNT_E("SEM_WAIT_FAILED IN X86_GRL_FONT_EXIT");
	}
}

void svgSetCacheableGlyphCount(SVGFontContext *ctx,
		  	  	  	  	  	  SVGUint32      glyph_count,
		  	  	  	  	  	  SVGFontInfo    *info)
{
    SVG_FNT_D("ENTERED INTO SVGUPDATEFONTS");
    if (local_is_valid_ctx(ctx, SVGSETCACHABLEGLYPHCOUNT) == SVG_TRUE)
    {
        SVG_FONT_WAI_LOCK
        ctx->p_grl_fp_table->SetCachableGlyphCount(ctx, glyph_count, info);
        SVG_FONT_SIG_LOCK
    }
    else
    {
        SVG_FNT_W("Use API only when valid font is loaded!");
    }
    SVG_FNT_D("EXITING FROM SVGINITFONTS");
}

SVGBoolean svgEnableShapingEng(SVGLangScript langscript,
                               SVGBoolean    autolang_detect_enable)
{
    SVGBoolean ret;

    SVG_FNT_D("ENTERED INTO SVGENABLESHAPINGENG");


    SVG_FONT_WAI_LOCK
    ret = grlftEnableShapingEng(langscript,autolang_detect_enable);
    SVG_FONT_SIG_LOCK

    SVG_FNT_D("EXITING FROM SVGENABLESHAPINGENG");

    return ret;
}


SVGBoolean svgDisableShapingEng(SVGLangScript langscript,
                                  SVGBoolean    autolang_detect_disable)
{
    SVGBoolean ret;

    SVG_FNT_D("ENTERED INTO SVGDISABLESHAPINGENG");

    SVG_FONT_WAI_LOCK
    ret = grlftDisableShapingEng(langscript,autolang_detect_disable);
    SVG_FONT_SIG_LOCK

    SVG_FNT_D("EXITING FROM SVGDISABLESHAPINGENG");

    return ret;
}
